<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;

use App\Newsletter;
use Illuminate\Http\Request;
use Illuminate\Support\Carbon;
use Illuminate\Support\Str;

class NewsletterController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $this->authorize('access','view_newsletters');
        $keyword = $request->get('search');
        $perPage = 25;

        if (!empty($keyword)) {

            $newsletters = Newsletter::whereFullText('subject',$keyword)->paginate($perPage);
        } else {
            $newsletters = Newsletter::orderBy('send_date','desc')->paginate($perPage);
        }

        return view('admin.newsletters.index', compact('newsletters'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $this->authorize('access','create_newsletter');
        return view('admin.newsletters.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        $this->authorize('access','create_newsletter');
        $this->validate($request,[
           'subject'=>['required'],
           'content'=>['required'],
           'send_date'=>['required']
        ]);

        if(array_sum([$request->mailing_list,$request->candidates,$request->employers])==0){
            return back()->withInput()->with('flash_message',__('site.select-recipient'));
        }


        $requestData = $request->all();

        $requestData['content'] = saveInlineImages($requestData['content']);
        $requestData['sent'] =  0;

        Newsletter::create($requestData);

        return redirect('admin/newsletters')->with('flash_message', __('site.changes-saved'));
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     *
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $this->authorize('access','view_newsletter');
        $newsletter = Newsletter::findOrFail($id);
        $hash  = Str::random(30);
        return view('admin.newsletters.show', compact('newsletter','hash'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     *
     * @return \Illuminate\View\View
     */
    public function edit($id)
    {
        $this->authorize('access','edit_newsletter');
        $newsletter = Newsletter::findOrFail($id);
        $newsletter->send_date = Carbon::parse($newsletter->send_date)->format('Y-m-d');

        return view('admin.newsletters.edit', compact('newsletter'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param  int  $id
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function update(Request $request, $id)
    {
        $this->authorize('access','edit_newsletter');
        $this->validate($request,[
            'subject'=>['required'],
            'content'=>['required'],
            'send_date'=>['required']
        ]);

        if(array_sum([$request->mailing_list,$request->candidates,$request->employers])==0){
            return back()->withInput()->with('flash_message',__('site.select-recipient'));
        }
        $requestData = $request->all();
        $requestData['content'] = saveInlineImages($requestData['content']);
        $newsletter = Newsletter::findOrFail($id);
        $newsletter->update($requestData);

        return redirect('admin/newsletters')->with('flash_message', __('site.changes-saved'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function destroy($id)
    {
        $this->authorize('access','delete_newsletter');
        Newsletter::destroy($id);

        return redirect('admin/newsletters')->with('flash_message', __('site.record-deleted'));
    }
}
