<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;

use App\MailingList;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Illuminate\Validation\Rule;

class MailingListController extends Controller
{



    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\View\View
     */
    public function index(Request $request)
    {
        $this->authorize('access','manage_mailing_list');
        $keyword = $request->get('search');
        $perPage = 50;

        if (!empty($keyword)) {
            $mailinglists = MailingList::where('email','LIKE','%'.$keyword.'%')->paginate($perPage);

        } else {
            $mailinglists = MailingList::latest()->paginate($perPage);
        }

        return view('admin.mailing-lists.index', compact('mailinglists'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\View\View
     */
    public function create()
    {
        $this->authorize('access','manage_mailing_list');
        return view('admin.mailing-lists.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function store(Request $request)
    {
        $this->authorize('access','manage_mailing_list');
        $this->validate($request,[
            'email'=>'required|unique:mailing_lists'
        ]);

        $requestData = $request->all();
        clearUnsubscribe($request->email);
        MailingList::create($requestData);

        return redirect('admin/mailing-lists')->with('flash_message', __('site.changes-saved'));
    }

    public function import(Request $request)
    {
        $this->authorize('access','manage_mailing_list');

        $emails = $request->emails;
        $emailArray = explode(PHP_EOL,$emails);
        $total = 0;
        foreach($emailArray as $email){
            $email = trim($email);
            $validator = Validator::make(['email'=>$email],[
                'email'=>['required','email','unique:mailing_lists']
            ]);

            if(!$validator->fails()){
                clearUnsubscribe($email);
                MailingList::create(['email'=>$email]);
                $total++;
            }
        }

        return redirect('admin/mailing-lists')->with('flash_message', __('site.emails-imported',['total'=>$total]));
    }

    public function deleteMultiple(Request $request){
        $this->authorize('access','manage_mailing_list');
        $count = 0;

        $data = $request->all();

        foreach($data as $value){
            if(is_numeric($value)){
                $count++;
                MailingList::find($value)->delete();
            }
        }
        return back()->with('flash_message',$count.' '.__('site.deleted'));
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     *
     * @return \Illuminate\View\View
     */
    public function show($id)
    {
        $this->authorize('access','manage_mailing_list');
        $mailinglist = MailingList::findOrFail($id);

        return view('admin.mailing-lists.show', compact('mailinglist'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     *
     * @return \Illuminate\View\View
     */
    public function edit($id)
    {
        $this->authorize('access','manage_mailing_list');
        $mailinglist = MailingList::findOrFail($id);

        return view('admin.mailing-lists.edit', compact('mailinglist'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param  int  $id
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function update(Request $request, $id)
    {
        $this->authorize('access','manage_mailing_list');
        $this->validate($request,[
            'email'=>[
                'required',
                Rule::unique('mailing_lists')->ignore($id),
            ]
        ]);
        $requestData = $request->all();

        $mailinglist = MailingList::findOrFail($id);
        $mailinglist->update($requestData);

        return redirect('admin/mailing-lists')->with('flash_message', __('site.changes-saved'));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     *
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function destroy($id)
    {
        $this->authorize('access','manage_mailing_list');
        MailingList::destroy($id);

        return redirect('admin/mailing-lists')->with('flash_message', __('site.record-deleted'));
    }
}
